/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/at_tach2.c
 # RCS:         $Id: at_tach2.c,v 1.6 1998/07/20 22:59:23 ericb Exp $
 # Description: Host program for demonstating resample at tach with
 #		multiple E1432/33s
 # Created:     November 9, 1997
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that demonstrates order tracking across two E1432s
 # using the resample at tach mode. It will display the resampled time 
 # waveforms and, optionally, the time waveforms.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432/33 with a tachometer option AYD
 #	3. another E1432/33 
 #
 # Set the logical address of the E1432/33s to the address specified in the 
 # LA1 and LA2 defined below (default 8 and 9).
 #
 # Connect a tach signal to the Tach1 connector with a frequency range of
 # 1KHz to 20Khz and separate 1 volt PP signal with a frequency range of
 # RPM_LOW/60 to RPM_HIGH/60 to inputs.
 #
 # The higher the tach signal frequency, the more time is
 # taken processing tach pulses by the CPU; so the performance begins
 # degrading above 10KHz.  Reducing the number of active input channels
 # (set NUM_CHANS_1 and NUM_CHANS_2 to a lower number than is available from 
 # the modules) and the blocksize will improve performance.
 #
 # To use the special external triggering mode, uncomment "#define EXT_TRIG".
 # and apply a triggering signal to the Tach2 connector.  The resampling
 # will occur after the trigger signal.  Each block is triggered by the
 # occurance of this signal.
 #
 # If the module can not keep up in real time, it will skip points
 # and issue a warning about lost points.  If the module can not process
 # tach pulses fast enough, it will overflow the internal raw tach buffer
 # and issue an error.
 #
 # The blocksize is set to NUM_REVS * PPR, or the number of revolutions of
 # the tach times the pulses/rev of the tach signal.  The inputs signals are
 # resampled at the frequency of the signal into the Tach1 connector.
 #
 ##############################################################################
 */
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include "e1432.h"
#include "err1432.h"
#include "xplot.h"

/*#define EXT_TRIG*/
/*#define TIME*/

#define NUM_MODS	2
#define NUM_CHANS_1     8
#define NUM_CHANS_2     8
#define LA1		8
#define LA2		9
#define WIDTH           140
#define HEIGHT          80
#define WARNING_MAX	100

#define NUM_REVS	2
#define PPR		512
#define RANGE		2.0
#define RPM_HIGH	6000
#define	RPM_LOW		100

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif


int
main(void)
{
    int  i, j, nchan1, nchan2; 
    int  total_chan1, total_chan2, total_chans, num_tachs;
    SHORTSIZ16 status;
    LONGSIZ32 count;
    SHORTSIZ16 la[2] = {LA1, LA2};
    SHORTSIZ16 tach_list[2];
    SHORTSIZ16 total_list[34];
    SHORTSIZ16 all, inputs, tachs, tach0; 
#ifdef EXT_TRIG
    SHORTSIZ16 tach1;
#endif
    E1432ID hw;
    float rpm[2], span;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[32];
    FLOATSIZ32 *resampled[32];
    long points;
    char *plotid[64];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange;
    SHORTSIZ16 warning[WARNING_MAX]; 
    unsigned long warningCount;
    int checkWarnings; 
    int module;
    struct e1432_trailer trailer;
    SHORTSIZ16 meas_state;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    int blocksize;
    float max_orders;

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    for(i=0; i < NUM_MODS; i++)
    {
        (void) printf("Checking for E1432 with firmware at logical address %d ... ", 
							(int)la[i]);
        (void) fflush(stdout);
        error = e1432_get_hwconfig(1, &(la[i]), &cf);
        if (error)
        {
	    (void) printf("Not found.\n");
	    (void) printf("Installing firmware from %s into E1432 at la %d ...",
					       semabinloc, (int)la[i]);
	    (void) fflush(stdout);
	    error = e1432_install(1, &(la[i]), 0, semabinloc);
	    if (error)
	    {
	        (void) printf("\ne1432_install failed and returned error %s\n");
	        exit(0);
	    }
	    (void) printf("Done.\n");
        }
        else
        {
	    (void) printf("Found.\n");
        }
    }
    (void) printf("\n");
    
    CHECK(e1432_print_errors(1));

    CHECK(e1432_assign_channel_numbers(2, la, &hw));

    error = e1432_get_hwconfig(1, &la[0], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    total_chan1 = cf.input_chans;
    nchan1 = (cf.input_chans < NUM_CHANS_1) ? cf.input_chans : NUM_CHANS_1;
    
    error = e1432_get_hwconfig(1, &la[1], &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }

    total_chan2 = cf.input_chans;
    nchan2 = (cf.input_chans < NUM_CHANS_2) ? cf.input_chans : NUM_CHANS_2;

    total_chans = nchan1 + nchan2;
    printf("Using %d channels from module 1 and %d channels from module 2\n",
							nchan1, nchan2);

    tach0 = tach_list[0] = E1432_TACH_CHAN(1);
#ifdef EXT_TRIG
    num_tachs = 2;
    tach1 = tach_list[1] = E1432_TACH_CHAN(2);
#else
    num_tachs = 1;
#endif
    tachs = e1432_create_channel_group(hw, num_tachs, tach_list);
    if (tachs >= 0)
    {
	(void) printf("e1432_create_channel_group for tachs returned %d\n",
		   						tachs);
	exit(0);
    }

    /* Create channel group */
    for(i=0; i < nchan1; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1);
    for(; i < total_chans; i++)
	total_list[i] = E1432_INPUT_CHAN(i+1+(total_chan1 - nchan1));
    for(; i < total_chans + num_tachs; i++)
	total_list[i] = E1432_TACH_CHAN(i - total_chans + 1);
    
    inputs = e1432_create_channel_group(hw, total_chans, total_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n", inputs);
	exit(0);
    }

    all = e1432_create_channel_group(hw, total_chans + num_tachs, total_list);
    if (all >= 0)
    {
	(void) printf("e1432_create_channel_group all returned %d\n", all);
	exit(0);
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, all, 0x400));  
#endif

    max_orders = PPR / 5.12;
    blocksize = NUM_REVS * PPR;
    
    if(blocksize > 4096)
    {
	printf("blocksize required > 4096, lower max_orders or PPR\n");
	exit(1);
    }

    printf("max_orders = %g\n", max_orders);
    printf("blocksize = %d\n", blocksize);
    points = blocksize;

    span = 5000;

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, RANGE));

    CHECK(e1432_set_data_size(hw, all, E1432_DATA_SIZE_32));

    CHECK(e1432_set_data_mode(hw, all, E1432_DATA_MODE_OVERLAP_BLOCK));

    row = col = 0;
    id = 0;
    temp = (float)(points - 1);

    for(i=0; i < total_chans; i++) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Can't malloc data array of %d points\n", points);
          exit(0);
        }

	for(j=0; j < points; j++) 
	{
	   data[i][j] = 0.0;
	}

        resampled[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!resampled[i]) {
          (void) printf("Can't malloc resampled array of %d points\n", points);
          exit(0);
        }

	for(j=0; j < points; j++) 
	{
	   resampled[i][j] = 0.0;
	}

        CHECK(e1432_get_range(hw, total_list[i], &floatRange));

	/* standard HPUX kernal won't support enough xplot windows */
	if(id < 48)
	{
#ifdef TIME
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
            (void) sprintf(title, "Time %d", i + 1);

            plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
	    if(plotid[id] == NULL)
	        (void) printf("xplot_init_plot() failed\n");
#endif
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
            (void) sprintf(title, "Resamp %d", i + 1);

            plotid[id+8] = xplot_init_plot(resampled[i], points, temp, 
		floatRange, -floatRange, GENERIC_TRACE, geometry, title);

	    if(plotid[id+8] == NULL)
	        (void) printf("xplot_init_plot() failed\n");

#ifdef TIME
	    xplot_change_yautoscale(plotid[id], 0);
            xplot_set_xscale(plotid[id], (float)0.0, temp);
            xplot_set_yscale(plotid[id], floatRange, -floatRange);
            xplot_change_xlabel(plotid[id], "Samples");
            xplot_change_ylabel(plotid[id], "Volts");
            xplot_repaint(plotid[id]); 
#endif
            xplot_change_yautoscale(plotid[id+8], 0);
            xplot_set_xscale(plotid[id+8], (float)0.0, temp);
            xplot_set_yscale(plotid[id+8], floatRange, -floatRange);
            xplot_change_xlabel(plotid[id+8], "Samples");
            xplot_change_ylabel(plotid[id+8], "Volts");
            xplot_repaint(plotid[id+8]); 
              
            col++;
            id++;
            if((col % 8) == 0) 
	    {
                col = 0;
                row +=2;
                id += 8;
            }
	}
    }

    CHECK(e1432_set_blocksize(hw, all, blocksize));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.2));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.3));

    CHECK(e1432_set_rpm_low(hw, tachs, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, tachs, RPM_HIGH)); 

    CHECK(e1432_set_max_order(hw, all, max_orders)); 
    CHECK(e1432_set_calc_data(hw, all, E1432_DATA_RESAMP_TIME)); 
    CHECK(e1432_set_decimation_output(hw, all, E1432_MULTIPASS)); 
    CHECK(e1432_set_decimation_oversample(hw, all, 
					E1432_DECIMATION_OVERSAMPLE_ON));

    CHECK(e1432_set_trigger_master(hw, tach0, E1432_TRIGGER_MASTER_ON)); 
    CHECK(e1432_set_sample_mode(hw, all, E1432_RESAMP_AT_TACH)); 
    
    CHECK(e1432_set_arm_channel(hw, tach0, E1432_CHANNEL_ON));

#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, tach1, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_slope(hw, tach1, E1432_TRIGGER_SLOPE_NEG));
#else
    CHECK(e1432_set_trigger_channel(hw, tach0, E1432_CHANNEL_ON));
#endif

#if 1
    CHECK(e1432_set_trigger_delay(hw, tachs, -blocksize/2)); 
#else
    CHECK(e1432_set_trigger_delay(hw, tachs, 0)); 
#endif

#ifndef TIME
    CHECK(e1432_set_enable(hw, inputs,
		    E1432_ENABLE_TYPE_TIME, E1432_ENABLE_OFF));
#endif

    CHECK(e1432_set_tach_ppr(hw, tach0, PPR));
    CHECK(e1432_set_span(hw, all, span));
    CHECK(e1432_get_span(hw, all, &span));
    printf("span = %g\n", span);

    /* Start measurement */
    CHECK(e1432_init_measure(hw, all));

    (void)sleep(1);

    CHECK(e1432_get_current_rpm(hw, tach0, rpm));
    (void) printf("current rpm = %g\n", rpm[0]);

    for (;;)
    {
	do  /* Wait for block available and check for errors and warnings  */
	{
	    CHECK(e1432_get_meas_state(hw, all, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }

	    if(nchan2 > 0)
	    {
	        CHECK(e1432_send_tachs(hw, all, tach0, NULL, 0, &count));
	    }

	    CHECK(e1432_read_register(hw, tach0, 	
					E1432_IRQ_STATUS2_REG, &status));
	    
	    if(status & E1432_IRQ_TRIGGER)
	    {
	        CHECK(e1432_send_trigger(hw, all));
	    }

	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed in module 1\n");
		else
		    (void) printf("Fifo overflowed in module 1\n");
		exit(1);
	    }

	    /* check for warnings in second module */
	    module = 1;
	    if((checkWarnings = status & E1432_IRQ_MEAS_WARNING) == 0)
	    {
	        if(nchan2)
	        {
	            /* check for warnings in second module */
	            module = 2;
	            CHECK(e1432_read_register(hw, total_list[nchan1], 	
				    E1432_IRQ_STATUS2_REG, &status));
	            if(status & E1432_IRQ_MEAS_ERROR)
	            {
			if(status & E1432_STATUS2_TACH_OVERFLOW)
		    	    (void) printf("Tach buffer overflowed in module 2\n");
			else
		    	    (void) printf("Fifo overflowed in module 2\n");
	                exit(1);
	            }
	        }

	        checkWarnings = status & E1432_IRQ_MEAS_WARNING;
	    }

	    if(checkWarnings)
	    {
		/* read out all measurement warnings */
		do
		{
		    CHECK(e1432_get_meas_warning(hw, all, 
						 warning, WARNING_MAX,
						 &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("Module %d: %d Warning", 
							module, warningCount);
			if(warningCount > 1) 
			    (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		} while (warningCount > 0);
	    }

	}while(e1432_block_available(hw, all)  == 0); 

        if (status < 0) {
	    (void) printf("status error %d\n", status);
	    exit(0);
        }

#ifdef TIME
	for(i=0; i < total_chans; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, total_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
              exit(0);
            }
	}
#endif

	id = 0;
        for(i=0; i < total_chans ; i++) {
            error = e1432_read_float32_data(hw, total_list[i], 
			E1432_RESAMP_DATA, resampled[i], points, 
							&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
              exit(0);
            }
            if (count != points) {
              (void) printf("Actual count was %d\n", count);
              exit(0);
            }

	    /* standard HPUX kernal won't support enough xplot windows */
            if(id < 48)
            {
#ifdef TIME
                xplot_check_events(plotid[id]);
                xplot_data_update(plotid[id]);
#endif
                xplot_check_events(plotid[id+8]);
                xplot_data_update(plotid[id+8]);
            }

	    if((++id%8)==0) 
	        id += 8;
        }
    }
    /*NOTREACHED*/
    return 0;
}

